/**
 *
 * \file        console.c
 *
 * \brief       Implements a character-based command and response
 *              command-line interface.
 *
 * \author      loosely based on console.c in Cajita project
 *
 * \date        12/5/2007
 *
 * \note        This is a generic file.  There should be no product-specific
 *              nor processor-specific stuff in here.
 *
 * to do:
 *
 * basic console
 * password support
 * help command
 */

////////////////////////////////////////////////////////////////////////////////

#include "console.h"
#include "os.h"
#include <stdarg.h>
#include "string_utils.h"
#include <string.h>
#include <ctype.h>
#include "errors.h"
#include "product.h"
#include <stdio.h>
#include "field_debug.h"
#include "dm_hardware.h"
#include "Cresnet.h"

#define UART_DABUG_CONSOLE_BUFFER_SIZE  81

////////////////////////////////////////////////////////////////////////////////

typedef struct
{
    UINT32 queueHandle;
    CMDINPUT * pCmdInput;               // current console cmd src
    INT32 debug;
    INT32 passwordEnabled;
    UINT8 extra[OS_TASK_EXTRA_BYTES];
}CONSOLE;

////////////////////////////////////////////////////////////////////////////////

static CONSOLE Console;

CMDINPUT * pReDirectCmdInput = NULL;

static UINT8 DmIsConsolePrintLocked( CMDINPUT * pCmdInput );
CMDTABLE *AltCommandTable = NULL;

struct DM_FIELD_DEBUG_STRUCT
{
    CMDINPUT* pCmdInput;
    UINT32 lSavedSendParam;
};
DM_FIELD_DEBUG_STRUCT g_FieldDebugParam = { NULL, 0 };

////////////////////////////////////////////////////////////////////////////////

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       see if the current command should be prohibited based on its source
 * \return      INT32
 * \retval      returns 0 meaning caller not banned
 * \param       CmdIndex, char caller
 */
static INT32 ConsoleCheckCallerBanned(int CmdIndex, int caller)
{
    // check for source
    if ( (CommandTable[CmdIndex].caller_banned & SRC_BANNED_MASK) == ANY_EXE )
	{
	    return 0;
	}

    if ( CommandTable[CmdIndex].caller_banned & TELNET_EXE )  // check if telnet log-in
	{
	    if ( caller == FROM_TELNET_CLIENT )
		return -1;
	}
    if ( CommandTable[CmdIndex].caller_banned & CTP_EXE )  // check if telnet log-in
	{
	    if ( caller == FROM_CTP_CLIENT )
		return -1;
	}
    if ( CommandTable[CmdIndex].caller_banned & REDIRECT_EXE )  // check if not for redirect
	{
	    if ( caller == FROM_CNET_REDIRECT )
		return -1;
	}
    if ( CommandTable[CmdIndex].caller_banned & SERIAL_EXE )  // check if not for serial
	{
	    if ( caller == FROM_UART_CONSOLE )
		return -1;
	}
    if ( CommandTable[CmdIndex].caller_banned & USB_EXE )  // check if not for USB
	{
	    if ( caller == FROM_USB_CONSOLE )
		return -1;
	}
    if ( CommandTable[CmdIndex].caller_banned & TJI_EXE )  // check if not for TJI
	{
	    if ( caller == FROM_TJI_REDIRECT )
		return -1;
	}
    if ( CommandTable[CmdIndex].caller_banned & PC_EXE )  // check if not for main (PC redirect)
	{
	    if ( caller == FROM_PC_CONSOLE )
		return -1;
	}

    return 0;
}

static INT32 ConsoleGetCommandNameLen (char *pcmd, UINT32 *pindex)
{
    int cnt, max_len;

    max_len = strlen(pcmd);
    for ( cnt = 0; cnt < max_len; cnt++ )
	{
	    if ( *(pcmd+cnt) == ' ' ||
		 *(pcmd+cnt) == '&' ||
		 *(pcmd+cnt) == 0x27 )
		{
		    break;
		}
	}

    *pindex = cnt;
    if ( *(pcmd+cnt) != 0 )
	{
	    *pindex += 1;
	}

    return(cnt);
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       ShowHelpMenu
 * \return      void
 * \retval      none
 * \param       MenuType
 */
void ShowHelpMenu(UINT32 MenuType)
{
    int idx ;
    char CmdName[20],*TmpPtr;


    if ( SYSTEM_TEST_HELP != MenuType )
	{
	    if ( MenuType & MAIN_HELP )
		{
		    // use the following to keep the same spacing
		    DmConsolePrintf("%-20.20s   %-40.40s\r", "?",
				    (MenuType == MAIN_HELP) ? "This help menu.": "Main Help screen" );
		}

	    for ( idx = 0 ; CommandTable[idx].match > 0 ; idx++ )
		{
		    if ( !(CommandTable[idx].help_menu & MenuType) )  // not this menu
			continue;

		    // see if allowed for this model.
		    //        if ( (CommandTable[idx].caller_banned & GetModelType() ) != 0 )
		    //            continue;

		    // only display the necessary chars for a match in UPPERCASE
		    strcpy(CmdName,CommandTable[idx].name);
		    TmpPtr = CmdName + CommandTable[idx].match;
		    while ( *TmpPtr != 0 )
			{
			    *TmpPtr = tolower(*TmpPtr);
			    TmpPtr++;
			}
		    // if command cannot be done for this connection, ignore it
		    if ( CommandTable[idx].name[0] > 0x20 )
			{
			    DmConsolePrintf("%-20.20s   %-40.40s\r",
					    (char *)CmdName,
					    (char *)&(CommandTable[idx].help_str[0]) ) ;
			}
		}
	}
    //print all commands from both tables with no help
    else
	{
	    //start with the commandtable
	    for ( idx = 0 ; CommandTable[idx].match > 0 ; idx++ )
		{
		    strcpy(CmdName,CommandTable[idx].name);

		    // if command cannot be done for this connection, ignore it
		    if ( CommandTable[idx].name[0] > 0x20 )
			DmConsolePrintf( "%s\r", (char *)CmdName ) ;
		}
	    //now the debugcommandtable
	    for ( idx = 0 ; DebugCommandTable[idx].match > 0 ; idx++ )
		{
		    strcpy(CmdName,DebugCommandTable[idx].name);

		    // if command cannot be done for this connection, ignore it
		    if ( DebugCommandTable[idx].name[0] > 0x20 )
			DmConsolePrintf( "%s\r", (char *)CmdName ) ;
		}

	}
}

INT8 ProcessCommand(CMDINPUT *pCmdInput, CMDTABLE *cmdTable, char *cmdPtr, INT32 *result)
{
    UINT32 idx;
    UINT32 cnt;
    UINT32 namelen;

    namelen = ConsoleGetCommandNameLen(cmdPtr, &cnt);

    for ( idx=0; cmdTable[idx].match > 0; idx++ )
	{
	    if ( !strncmp(cmdTable[idx].name, cmdPtr,
			  (cmdTable[idx].match>namelen) ? cmdTable[idx].match : namelen) )
		{
		    break;
		}
	}

    if ( cmdTable[idx].match != 0 ) // match found
	{
	    // found a match
	    *result = ConsoleCheckCallerBanned(idx, pCmdInput->cmdSource);
	    if ( *result == -1 )
		{
		    DmConsolePrintf("Command Blocked");
		    DmConsoleShowPrompt();
		    return(-1);
		}

	    // command is not banned
	    if ( !cmdTable[idx].func_call )
		{
		    // no command handler
		    DmConsoleShowPrompt();
		    return(-1);
		}

	    // call the command handler
	    *result = cmdTable[idx].func_call(cmdTable[idx].level, (char *)(&cmdPtr[cnt]));
	    return(1);
	}
    return(0);
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       This gets called once for every command that gets parsed
 * \return      void
 * \retval      none
 * \param       param - pointer to command struction
 */
void ConsoleTask(UINT32 param)
{
    CMDSTRUCT * cmd = (CMDSTRUCT *)param;

    char * cmdPtr;
    CMDINPUT * pCmdInput = (CMDINPUT *)cmd->pCmdInput;
    //char * tmpPtr;
    INT32 result = 0;
    UINT32 idx;
    UINT32 cnt;
    UINT32 namelen;
    INT32 CommandFound;

    // if the command is not from the console
    if ( pCmdInput->cmdSource != Console.pCmdInput->cmdSource )
        // redirect the output to this sources tx
        // (will be cleared after prompt is sent)
        pReDirectCmdInput = pCmdInput;
    else
        pReDirectCmdInput = NULL;

    cmdPtr = cmd->commandString;

    // strip off leading white space
    while ( (isspace(*cmdPtr)) &&
            (*cmdPtr != 0) &&
            (*cmdPtr != EOL_CHAR) )
	{
	    cmdPtr++;
	}

    // just turn echo on
    if ( cmd->length == 1 )
	{
	    pCmdInput->echo = 1;
	    DmConsoleShowPrompt();
	    return;
	}

    if ( Console.passwordEnabled )
	{
	    // do later
	}

    DmConsolePrintf("\r");

    if ( cmdPtr[0] == '?' )
	{
	    ShowHelpMenu(MAIN_HELP);
	    DmConsoleShowPrompt();
	    return;
	}

    result = 0;
    LocalConvertToUpper(cmdPtr);
    namelen = ConsoleGetCommandNameLen(cmdPtr, &cnt);

    //If we got a .XA[LF], send te welcome message
    if (namelen == 3 && cmdPtr[0] == 0x1B && cmdPtr[1] == 0x58 && cmdPtr[2] == 0x41)
	{
	    DmConsoleDisplayWelcome(pCmdInput);
	    return;
	}

    CommandFound = ProcessCommand(pCmdInput, (CMDTABLE *)CommandTable, cmdPtr, &result);

    if ((!CommandFound) && (AltCommandTable))
	CommandFound = ProcessCommand(pCmdInput, (CMDTABLE *)AltCommandTable, cmdPtr, &result); 

    //command not found in either primary or alternate command tables

    if (!CommandFound)
	
	{
	    //check the debug command list
	    for ( idx=0; DebugCommandTable[idx].match > 0; idx++ )
		{
		    if ( !strncmp(DebugCommandTable[idx].name, cmdPtr, 
				  (DebugCommandTable[idx].match>namelen) ? 
				  DebugCommandTable[idx].match : namelen) )
			{
			    break;
			}
		}

	    //if not found here either, print debug message and return
	    if ( DebugCommandTable[idx].match == 0 )
		{
		    DmConsolePrintf("Bad or Incomplete Command");
		    DmConsoleShowPrompt();
		    return;
		}

	    //if we got here debug command was found
	    if ( !DebugCommandTable[idx].func_call )
		{
		    // no command handler
		    DmConsoleShowPrompt();
		    return;
		}

	    // call the command handler
	    result = DebugCommandTable[idx].func_call( 0, (char *)(&cmdPtr[cnt]));
	}
    if (( result != 0 )&&( result != INHIBIT_PROMPT ))
	{
	    DmConsolePrintf("ERROR: Console command returned error code %d\r", result);
	}

    if ( result == -999 )
	{
	    // command is requesting death of console
	    pCmdInput->killMe = 1;
	}

    if ( result != INHIBIT_PROMPT )
	{
	    DmConsoleShowPrompt() ;
	}
    else                                   //If prompt is not showed, console is not unlocked, because unlock occurs in prompt.
	{
	    pCmdInput->consoleLock = 0;
	}

}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       DmConsoleInit
 * \return      void
 * \retval      none
 * \param       void
 */
void DmConsoleInit(UINT32 lStackSize)
{
    UINT32 param = 0;

    if ( lStackSize == 0 )
	lStackSize = OsGetDefaultStackVarCnt();
    // entry, periodic, waiter, param, extra
    ///\todo Revisit the priority of this task
    OsCreateNamedPriorityTask(ConsoleTask, 0, DM_CONSOLE_QUEUE_COUNT_MAX, &param, Console.extra, "Con", CONSOLE_TASK_PRIORITY, lStackSize);

    // need this so we can send commands to it
    Console.queueHandle = param;
}

// add password support later
/**
 * \author      Pete McCormick, Larry Salant
 * \date        3/12/2008
 * \brief       DmConsoleShowPrompt
 * \return      void
 * \retval      none
 * \param       void
 */
void DmConsoleShowPrompt(void)
{
    CMDINPUT * pCmdInput = NULL;

    // if the console is redirected
    if ( pReDirectCmdInput != NULL )
	pCmdInput = pReDirectCmdInput;     //use the sources output
    else if ( Console.pCmdInput != NULL ) //use the consoles, if they exist
	pCmdInput = Console.pCmdInput;

    if ( pCmdInput != NULL )
	// indicate buffer not in use
	pCmdInput->consoleLock = 0;

    DmConsolePrintf(DmConsoleGetPrompt());

    // end redirection, if any
    pReDirectCmdInput = NULL;
}

/*
 * \author      Gennady
 * \date        8/20/2008
 * \brief       print string of any size to the current console
 * \return      void
 * \retval      none
 * \param       format
 */
void DmConsoleNPuts(char *src, UINT16 length)
{
    CMDINPUT * pCmdInput;
    UINT32 timeout = MAX_CONSOLE_WAIT;//2 seconds (1000x2)

    if ( InIsr )
	return;

    // if the console is redirected
    if ( pReDirectCmdInput != NULL )
	pCmdInput = pReDirectCmdInput;     //use the sources output
    else if ( Console.pCmdInput != NULL ) //use the consoles, if they exist
	pCmdInput = Console.pCmdInput;
    else
	return;

    //Bugzilla 80918 fix
    //Skip print if the RCON packet has the "no display" flag
    if( (pCmdInput->cmdSource == RCON_EXE) && (pCmdInput->sendParam & RCON_CMD_OPTION_NO_DISPLAY) )
    {
        return;// do not display a response
    }

    if ( pCmdInput->captured && DmIsConsolePrintLocked(pCmdInput) )
	{
#ifdef DM_CONTROLLER_UPLOAD_DEBUG
	    DmRebootNotFatal();
#endif
	    return;
	}

    // wait for lock available
    while ( timeout > 0 )
	{
	    // turn off interrupts
	    OsEnterCritical();
	    // if no one is printing
	    if ( !DmIsConsolePrintLocked(pCmdInput) )
		{
		    // set printing in use flag
		    pCmdInput->printLock = 1;
		    // interrupts on
		    OsExitCritical();
		    break;
		}

	    // interrupts on
	    OsExitCritical();
	    HwDelayMsec(2);
	    timeout--;
	}

    //  if we got the lock
    if ( timeout != 0 )
	{
	    pCmdInput->pSendFunc((unsigned char *)src, length, pCmdInput->sendParam);
	    pCmdInput->printLock = 0;
	}

    return;
}

void DmConsolePuts(char *src)
{
    DmConsoleNPuts(src, strlen((char const *)src));
}

void catNumToStr( char * pDest, UINT16 num )
{
    char pAsciiNum[2]; //ascii num and terminator
    pAsciiNum[1] = 0;     //terminator
    bool nonzero = false;

    UINT16 div = 10000;
    UINT16 digit;// = num/div;

    while ( div )
	{
	    digit = num/div;

	    if ( digit || nonzero || div==1 )
		{
		    nonzero = true;
		    pAsciiNum[0] = 0x30 + digit;
		    strcat( pDest, pAsciiNum );
		    num -= div*digit;
		}

	    div /= 10;
	}
}

void DmConsolePrintError( UINT8 level, UINT8 subsystem, UINT8 errCode, UINT16 causeCode )
{
    char * pDest;
    CMDINPUT * pCmdInput;
    int str_len=0;
    UINT32 timeout = MAX_CONSOLE_PRINT_WAIT;//250 ms

    if ( InIsr )
	return;

    // if the console is redirected
    if ( pReDirectCmdInput != NULL )
	pCmdInput = pReDirectCmdInput;     //use the sources output
    else if ( Console.pCmdInput != NULL ) //use the consoles, if they exist
	pCmdInput = Console.pCmdInput;
    else
	return;

    //Bugzilla 80918 fix
    //Skip print if the RCON packet has the "no display" flag
    if( (pCmdInput->cmdSource == RCON_EXE) && (pCmdInput->sendParam & RCON_CMD_OPTION_NO_DISPLAY) )
    {
        return;// do not display a response
    }

    if ( pCmdInput->captured && DmIsConsolePrintLocked(pCmdInput) )
	{
#ifdef DM_CONTROLLER_UPLOAD_DEBUG
	    DmRebootNotFatal();
#endif
	    return;
	}

    // wait for lock available
    while ( timeout > 0 )
	{
	    // turn off interrupts
	    OsEnterCritical();
	    // if no one is printing
	    if ( !DmIsConsolePrintLocked(pCmdInput) )
		{
		    // set printing in use flag
		    pCmdInput->printLock = 1;
		    // interrupts on
		    OsExitCritical();
		    break;
		}

	    // interrupts on
	    OsExitCritical();
	    HwDelayMsec(1);
	    timeout--;
	}

    //  if we got the lock
    if ( timeout != 0 )
	{
	    // print
	    pDest = (char *)pCmdInput->txBuf;

	    strcpy( pDest, "ERR ");

	    switch ( level )
		{
		case DM_ERROR_LEVEL_FATAL:
		    strcat( pDest, "FATAL" );
		    break;
		case DM_ERROR_LEVEL_ERROR:
		    strcat( pDest, "ERROR" );
		    break;
		case DM_ERROR_LEVEL_WARNING:
		    strcat( pDest, "WARNING" );
		    break;
		case DM_ERROR_LEVEL_NOTICE:
		    strcat( pDest, "NOTICE" );
		    break;
		case DM_ERROR_LEVEL_DEBUG:
		    strcat( pDest, "DEBUG" );
		    break;
		default:
		    break;
		}

	    strcat( pDest, ": subsystem ");
	    catNumToStr( pDest, subsystem );
	    strcat( pDest, ", errorcode ");
	    catNumToStr( pDest, errCode );
	    strcat( pDest, ", cause ");
	    catNumToStr( pDest, causeCode );
	    strcat( pDest, "\r\n");

	    str_len = strlen(pDest);

	    if ( (pCmdInput->txBuf[str_len-1] == 0xA) && (pCmdInput->txBuf[str_len-2] != 0xD) )
		pCmdInput->txBuf[str_len-1] = 0xD;
	    if ( (pCmdInput->txBuf[str_len-2] == 0xA) && (pCmdInput->txBuf[str_len-1] != 0xD) )
		pCmdInput->txBuf[str_len-2] = 0xD;

	    pCmdInput->pSendFunc((UINT8*)pDest, str_len, pCmdInput->sendParam);

	    // clear the lock
	    pCmdInput->printLock = 0;
	}
    return;
}

/**
 * \author      Pete McCormick, Larry Salant
 * \date        3/12/2008
 * \brief       print something to the current console
 * \return      void
 * \retval      none
 * \param       format
 */
void DmConsolePrintf(char *format, ...)
{
    char * pDest;
    va_list ap;
    CMDINPUT * pCmdInput = NULL;
    int str_len=0;
    UINT32 timeout = MAX_CONSOLE_PRINT_WAIT;//250 ms

    if (InIsr)
	return;

    // if the console is redirected
    if ( pReDirectCmdInput != NULL )
	pCmdInput = pReDirectCmdInput;     //use the sources output
    else if ( Console.pCmdInput != NULL ) //use the consoles, if they exist
	pCmdInput = Console.pCmdInput;
    else
	return;

    //Bugzilla 80918 fix
    //Skip print if the RCON packet has the "no display" flag
    if( (pCmdInput->cmdSource == RCON_EXE) && (pCmdInput->sendParam & RCON_CMD_OPTION_NO_DISPLAY) )
    {
        return;// do not display a response
    }

    if ( pCmdInput->captured && DmIsConsolePrintLocked(pCmdInput) )
	{
#ifdef DM_CONTROLLER_UPLOAD_DEBUG
	    DmRebootNotFatal();
#endif
	    return;
	}

    // wait for lock available
    while ( timeout > 0 )
	{
	    // turn off interrupts
	    OsEnterCritical();
	    // if no one is printing
	    if ( !DmIsConsolePrintLocked(pCmdInput) )
		{
		    // set printing in use flag
		    pCmdInput->printLock = 1;
		    // interrupts on
		    OsExitCritical();
		    break;
		}

	    // interrupts on
	    OsExitCritical();
	    HwDelayMsec(1);
	    timeout--;
	}

    //  if we got the lock
    if ( timeout != 0 )
	{
	    // print
	    pDest = (char *)pCmdInput->txBuf;
	    // PEM - use vsnprintf instead of vsprintf for safety!
	    va_start(ap, format);
	    vsnprintf(pDest, sizeof(pCmdInput->txBuf), format, ap);
	    va_end(ap);

	    str_len = strlen(pDest);

	    if ( (pCmdInput->txBuf[str_len-1] == 0xA) && (pCmdInput->txBuf[str_len-2] != 0xD) )
		pCmdInput->txBuf[str_len-1] = 0xD;
	    if ( (pCmdInput->txBuf[str_len-2] == 0xA) && (pCmdInput->txBuf[str_len-1] != 0xD) )
		pCmdInput->txBuf[str_len-2] = 0xD;

	    pCmdInput->pSendFunc((UINT8*)pDest, str_len, pCmdInput->sendParam);

	    // clear the lock
	    pCmdInput->printLock = 0;
	}
    return;
}

/**
 * \author      Adolfo Velasco
 * \date        12/18/2013
 * \brief       Debug print that uses CMDINPUT pointer saved when FDEBUG was enabled. 
 *              This makes it immune to extra RCON commands that change the console 
 * \return      void
 * \retval      none
 * \param       format
 */
void DmDebugPrintf(char *format, ...)
{
    char * pDest;
    va_list ap;
    CMDINPUT * pCmdInput = NULL;
    int str_len=0;
    UINT32 timeout = MAX_CONSOLE_PRINT_WAIT;//250 ms
    UINT32 lSendParam = 0;

    if (InIsr)
	return;

    if ( g_FieldDebugParam.pCmdInput )
	{
        //Bugzilla fix
        //If g_FieldDebugParam has a valid pointer, utilize the pointer and saved send parameter
	    pCmdInput = g_FieldDebugParam.pCmdInput;
        lSendParam = g_FieldDebugParam.lSavedSendParam;
	}
    else
	{
        //Bugzilla fix
        //If no field debug parameter information was saved, use the current console pointer and current send parameter if possible
	    pCmdInput = Console.pCmdInput;
        if ( Console.pCmdInput )
        {
            lSendParam = Console.pCmdInput->sendParam;
        }
	}

    if ( !pCmdInput )
        return;

    //Bugzilla 80918 fix
    //Skip print if the RCON packet has the "no display" flag
    if( (pCmdInput->cmdSource == RCON_EXE) && (lSendParam & RCON_CMD_OPTION_NO_DISPLAY) )
    {
        return;// do not display a response
    }

    if ( pCmdInput->captured && DmIsConsolePrintLocked(pCmdInput) )
	{
	    return;
	}

    // wait for lock available
    while ( timeout > 0 )
	{
	    // turn off interrupts
	    OsEnterCritical();
	    // if no one is printing
	    if ( !DmIsConsolePrintLocked(pCmdInput) )
		{
		    // set printing in use flag
		    pCmdInput->printLock = 1;
		    // interrupts on
		    OsExitCritical();
		    break;
		}

	    // interrupts on
	    OsExitCritical();
	    HwDelayMsec(1);
	    timeout--;
	}

    //  if we got the lock
    if ( timeout != 0 )
	{
	    // print
	    pDest = (char *)pCmdInput->txBuf;
	    // PEM - use vsnprintf instead of vsprintf for safety!
	    va_start(ap, format);
	    vsnprintf(pDest, sizeof(pCmdInput->txBuf), format, ap);
	    va_end(ap);

	    str_len = strlen(pDest);

	    if ( (pCmdInput->txBuf[str_len-1] == 0xA) && (pCmdInput->txBuf[str_len-2] != 0xD) )
		pCmdInput->txBuf[str_len-1] = 0xD;
	    if ( (pCmdInput->txBuf[str_len-2] == 0xA) && (pCmdInput->txBuf[str_len-1] != 0xD) )
		pCmdInput->txBuf[str_len-2] = 0xD;

	    pCmdInput->pSendFunc((UINT8*)pDest, str_len, lSendParam);

	    // clear the lock
	    pCmdInput->printLock = 0;
	}
    return;
}

/**
 * \author      Adolfo Velasco
 * \date        12/19/2013
 * \brief       Updates the global field debug data parameters
 * \return      void
 * \retval      none
 * \param       format
 */
void UpdateFieldDebugParam( void )
{
    //Assume field debug is disabled
    UINT8 bFieldDebugEnabled = false;
    for ( UINT8 i = 0; i < DM_FIELD_DEBUG_ARRAY_SIZE; i++ )
    {
        if ( g_lDmFieldDebugState[i] > 0 )
        {
            bFieldDebugEnabled = true;
            break;
        }
    }

    //Check if field debug is enabled
    if ( bFieldDebugEnabled )
    {
        //Save the CMDINPUT pointer

        //Check and use a redirection console if possible
        if ( pReDirectCmdInput )
        {
            g_FieldDebugParam.pCmdInput = pReDirectCmdInput;            
        }
        //Check and use the current console next if possible
        else if ( Console.pCmdInput )
        {
            g_FieldDebugParam.pCmdInput = Console.pCmdInput;            
        }
        else
        {
            g_FieldDebugParam.pCmdInput = NULL;            
        }

        //Save the parameter (in case someone tries to block RCON echo)
        if ( g_FieldDebugParam.pCmdInput )
        {
            g_FieldDebugParam.lSavedSendParam = g_FieldDebugParam.pCmdInput->sendParam;
        }
    }
    //Field debug is disabled
    else
    {
        //Clear the parameters
        g_FieldDebugParam.pCmdInput = NULL;
        g_FieldDebugParam.lSavedSendParam = 0;
    }
}

/**
 * \author      Dariusz Rymsza
 * \date        2/14/2013
 * \brief       print something to the UART console
 * \brief       To save memory, no locks are used, not thread
 * \brief       safe
 * \return      void
 * \retval      none
 * \param       format
 */
void DmUARTConsolePrintf(char *format, ...)
{
    char * pDest;
    va_list ap;
    int str_len=0;

    if (InIsr)
	return;

    // print
    pDest = (char *)malloc(UART_DABUG_CONSOLE_BUFFER_SIZE);
    if ( !pDest )
	return;

    // PEM - use vsnprintf instead of vsprintf for safety!
    va_start(ap, format);
    vsnprintf(pDest, UART_DABUG_CONSOLE_BUFFER_SIZE - 1, format, ap);
    va_end(ap);

    str_len = strlen(pDest);

    if ( (pDest[str_len-1] == 0xA) && (pDest[str_len-2] != 0xD) )
	pDest[str_len-1] = 0xD;
    if ( (pDest[str_len-2] == 0xA) && (pDest[str_len-1] != 0xD) )
	pDest[str_len-2] = 0xD;

    UartConsoleTx((UINT8*)pDest, strlen(pDest), HwGetConsoleUart());
}

// CmdInputCreate() has been replaced by just filling in a struct

void DmConsoleHexDumpPacket(UINT8 * pPkt, UINT16 byteCnt, char * label,
			    BOOL showAscii)
{
    INT32 i;
    int val;
    INT32 j;
    INT32 k;

    if ( !byteCnt )
	{
	    return;
	}

    DmConsolePrintf("\r\n%s", label);

    for ( i=0; i<byteCnt; i++ )
	{
	    if ( i % 16 == 0 )
		{
		    DmConsolePrintf("\r");
		}

	    val = pPkt[i] & 0xff;
	    DmConsolePrintf("%02.2x ", val);

	    if ( (i % 16 == 15) || (i == byteCnt-1) )
		{
		    if ( i == byteCnt-1 )
			{
			    // print spaces for missing data elements
			    for ( j=0; j<15-(i%16); j++ )
				{
				    DmConsolePrintf("   ");
				}
			}

		    if ( showAscii )
			{
			    for ( j=0; j<=i%16; j++ )
				{
				    k = (i-i%16) + j;
				    if ( k >= 0 )
					{
					    if ( (pPkt[k] >= ' ') && (pPkt[k] <= '~') )
						{
						    DmConsolePrintf("%c", pPkt[k]);
						}
					    else
						{
						    DmConsolePrintf(".");
						}
					}
				    else
					{
					    DmConsolePrintf(".");
					}
				}
			}
		}
	}

    DmConsolePrintf("\r");
}

void ConsoleRestoreConfig(void)
{
}

void DmConsoleInitDefaults(void)
{
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       DmConsoleSendCommandToTask
 * \return      INT32
 * \retval      success
 * \param       CMDSTRUCT * pCommand
 */
INT32 DmConsoleSendCommandToTask(CMDSTRUCT * pCommand)
{
    INT32 status;

    CMDINPUT * pCmdInput = (CMDINPUT *)pCommand->pCmdInput;
    // lock the buffer until command is processed
    pCmdInput->consoleLock = 1;

    // put the command on the console task's queue
    if (( status = OsQueue(Console.queueHandle, (void *)&pCommand, CONSOLE_QUEUE_TIMEOUT )) == -1)
	{
	    //  pCommand->cmdSource:
	    //  FROM_CTP_CLIENT                 0x0002
	    //  FROM_UART_CONSOLE               0x0008
	    //  FROM_USB_CONSOLE                0x0020
	    //  TJI_EXE                         0x0040
	    //  RCON_EXE                        0x0080
	    //  PASSTO_EXE                      0x0100

	    DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_GENERAL,ERR_GENERAL_CONSOLE_QUEUE_TIMEOUT,((CMDINPUT *)&pCommand->pCmdInput)->cmdSource);

        //Clear the lock since the event was not queued properly
        pCmdInput->consoleLock = 0;
	}

    return status;

}

// add the specified cmd input to a global list
// of possible command inputs
/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       CmdInputAddToList
 * \return      void
 * \retval      none
 * \param       CMDINPUT *pCmdInput
 */
void CmdInputAddToList(CMDINPUT *pCmdInput)
{
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       DmConsoleDisplayWelcome
 * \return      void
 * \retval      none
 * \param       CMDINPUT * destination
 */
void DmConsoleDisplayWelcome(CMDINPUT * destination)
{
    Console.pCmdInput = destination;

    // end redirection, if any
    pReDirectCmdInput = NULL;

    DmConsolePrintf(ProductGetWelcomeMsg());
    DmConsoleShowPrompt();
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       just buffers a line until a CR, then sends to console task
 * \return      UINT32
 * \retval      returns OS_ISR_FLAG_TASK_WOKEN if something woke up
 * \param       CMDINPUT * destination
 */
UINT32 DmConsoleBufferInput(CMDINPUT * pCmdInput, UINT8 newChar)
{
    UINT8 echoChar;
    //INT32 result;
    UINT32 flags = 0;

    if ( pCmdInput->echo )
	{
	    echoChar = newChar;
	    // to do: support password
	    //if ( (EnteringPassword & CNET_SETTING_PASSWORD) &&
	    //        (EchoChar != ASC_BKSP) && (EchoChar != ASC_CR) )
	    //    EchoChar = '*';  // don't display password

	    // echo the character if it is not a backspace,
	    // unless there are some characters on the line to erase
	    if ( (echoChar != ASC_BKSP) || (pCmdInput->rxBufBytes > 0) )
		{
		    //SendCresnetConsoleMsg((UINT8 *)&EchoChar,1);
		    pCmdInput->pSendFunc(&echoChar, 1, pCmdInput->sendParam);
		}
	}

    if ( newChar == ASC_BKSP )
	{
	    if ( pCmdInput->rxBufBytes != 0 )
		{
		    pCmdInput->rxBufBytes--;
		}
	}
    else
	{
	    if ( pCmdInput->rxBufBytes >= DM_MAX_USER_CMD_LEN-1 )  // overflow
		{
		    // process the line anyway,
		    // so we don't get stuck forever
		    pCmdInput->rxBufBytes = DM_MAX_USER_CMD_LEN-1;
		    newChar = EOL_CHAR;
		}

	    if ( newChar >= 0x20 ||  newChar == 0x1B)  // pass printable characters and Escape
		{
		    pCmdInput->rxBuf[pCmdInput->rxBufBytes] = newChar;
		    pCmdInput->rxBufBytes++;
		}

	    if ( newChar == EOL_CHAR )
		{
		    pCmdInput->rxBuf[pCmdInput->rxBufBytes] = 0;
		    pCmdInput->rxBufBytes++;// include NULL

		    // got a complete message.  Send to Console task
		    _strncpy(pCmdInput->command.commandString, (char *)pCmdInput->rxBuf,
			     sizeof(pCmdInput->command.commandString));
		    pCmdInput->command.length = pCmdInput->rxBufBytes;
		    // command is sent by a pointer, so that's why we have to
		    // double-buffer here
		    // reset to begining of buffer
		    pCmdInput->rxBufBytes = 0;
		    if ( DmConsoleSendCommandToTask(&pCmdInput->command) == 1 )
			{
			    flags |= OS_ISR_FLAG_TASK_WOKEN;
			}
		}
	}
    return flags;
}


/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       DmConsoleIsLocal
 * \return      BOOL
 * \retval      none
 * \param       void
 */
BOOL DmConsoleIsLocal(void)
{
    return 1;
}

/**
 * \author      Pete McCormick
 * \date        3/12/2008
 * \brief       DmConsoleGetUserResponse
 * \return      void
 * \retval      none
 * \param       prompt
 */
void DmConsoleGetUserResponse(char *prompt, char *ResponseString, UINT32 MaxLength)
{
}

void DmConsoleSetEcho(BOOL echo)
{
    // if the console is redirected
    if ( pReDirectCmdInput != NULL )
	pReDirectCmdInput->echo = echo;
    else if ( Console.pCmdInput != NULL ) //use the consoles, if they exist
	Console.pCmdInput->echo = echo;
}

BOOL DmConsoleGetEcho(void)
{
    // if the console is redirected
    if ( pReDirectCmdInput != NULL )
	return pReDirectCmdInput->echo;
    else if ( Console.pCmdInput != NULL ) //use the consoles, if they exist
	return Console.pCmdInput->echo;
    return 1;
}

/**
 * \author      Larry  Salant
 * \date        10/16/2008
 * \brief       poor man's (memory limited) semaphore for the command input buffer
 * \return      UINT8
 * \retval      0 = timed out
 * \param       ptr to command input buffer
 * \note        CANNOT be called from Interrupt Handler
 */
BOOL DmConsoleWaitBufferAvailable(CMDINPUT * pCmdInput, UINT16 sTimeout)
{
    UINT16 sCount = sTimeout;

    while ( pCmdInput->consoleLock != 0 && sCount > 0 )
	{
	    HwDelayMsec(DM_CONSOLE_WAIT_POLL_RATE_MS);
	    sCount--;
	}

    //Check if we care about a timeout
    if ( sTimeout > 0 && sCount == 0 && pCmdInput->consoleLock != 0 )
    {
        DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_GENERAL,ERR_GENERAL_CONSOLE_TIMEOUT,0);
    }

    //Buffer available means console is unlocked
    return ( !pCmdInput->consoleLock ? TRUE : FALSE );
}

/**
 * \author      Andrew Salmon
 * \date        08/04/2011
 * \brief       poor man's (memory limited) semaphore for tji print
 * \return      UINT8
 * \retval      0 = timed out
 * \param       ptr to command input buffer
 * \note        CANNOT be called from Interrupt Handler
 */
UINT8 DmConsoleTjiPrintLock(void)
{
    UINT16 timeout       = MAX_CONSOLE_TJI_PRINT_WAIT;
    CMDINPUT * pCmdInput = NULL;

    if ( InIsr )
	return 0;

    // if the console is redirected
    if ( pReDirectCmdInput != NULL )
	pCmdInput = pReDirectCmdInput;     //use the sources output
    else if ( Console.pCmdInput != NULL ) //use the consoles, if they exist
	pCmdInput = Console.pCmdInput;

    if ( pCmdInput )
	{
	    while ( timeout > 0 )
		{
		    // if no one is printing Tji
		    if ( !pCmdInput->TjiPrintLock )
			{
			    // turn off interrupts
			    OsEnterCritical();
			    // set printing in use flag
			    pCmdInput->TjiPrintLock = (UINT32)OsGetCurrentTaskHandle();
			    // interrupts on
			    OsExitCritical();
			    return 1;//locked
			}

		    HwDelayMsec(MAX_CONSOLE_TJI_PRINT_WAIT_PERIOD);
		    timeout--;
		}
	}
    // if we timed out, report error
    if ( timeout == 0 || !pCmdInput )
	DmSystemError(DM_ERROR_LEVEL_ERROR,DM_ERROR_SUBSYS_GENERAL,ERR_GENERAL_CONSOLE_TJI_LOCK_FAILED,0);

    return 0;
}
/**
 * \author      Andrew Salmon
 * \date        08/04/2011
 * \brief       Allow new task to print tji to console
 * \return
 * \retval
 * \param
 * \note        CANNOT be called from Interrupt Handler
 */
void DmConsoleTjiPrintUnlock(void)
{
    CMDINPUT * pCmdInput = NULL;
    // turn off interrupts
    OsEnterCritical();
    // if the console is redirected
    if ( pReDirectCmdInput != NULL )
	pCmdInput = pReDirectCmdInput;     //use the sources output
    else if ( Console.pCmdInput != NULL ) //use the consoles, if they exist
	pCmdInput = Console.pCmdInput;

    if ( pCmdInput  )
	{
	    if (pCmdInput->TjiPrintLock == (UINT32)OsGetCurrentTaskHandle())
		pCmdInput->TjiPrintLock = 0;
	}

    // interrupts on
    OsExitCritical();
}
/**
 * \author      Andrew Salmon
 * \date        10/16/2008
 * \brief       Query if console is locked by another task
 * \return      UINT8
 * \retval      1 = unlocked for this task
 * \param       pCmdInput to command input buffer
 * \note
 */
static UINT8 DmIsConsolePrintLocked( CMDINPUT * pCmdInput )
{
    if ( pCmdInput )
	{
	    if ( pCmdInput->printLock || ( pCmdInput->TjiPrintLock && 
					   (pCmdInput->TjiPrintLock != 
					    (UINT32)OsGetCurrentTaskHandle()) ) )
		{
		    return 1;
		}
	}
    return 0;
}
/**
 * \author      Andrew  Salmon
 * \date        05/21/2009
 * \brief       Returns current console
 * \return      CMDINPUT
 * \retval      pointer to the console data
 *
 */
CMDINPUT * DmGetCurrentConsole( void )
{
    if ( pReDirectCmdInput != NULL )
	return pReDirectCmdInput;
    else
	return(Console.pCmdInput);
}

/**
 * \author      S.Novick
 * \date        04/22/2013
 * \brief       Kills selected console
 * \param       CMDINPUT
 * \return      void
 * \retval      none
 */
void DmConsoleCloseInput( CMDINPUT *pCmdInput )
{
    if ( pReDirectCmdInput == pCmdInput )
	pReDirectCmdInput = NULL;
    else if ( Console.pCmdInput == pCmdInput )
	Console.pCmdInput = NULL;
    else
	return;
}

/**
 * \author      S.Novick
 * \date        12/16/2013
 * \brief       Redirects current console
 * \param       CMDINPUT
 * \return      void
 * \retval      none
 */
void DmConsolePassto(CMDINPUT *pCmdPassto)
{
    CMDINPUT* pCmdInput = DmGetCurrentConsole();

    if (pCmdPassto && (pCmdInput != pCmdPassto))
	{
	    pCmdInput->pPassto = pCmdPassto;
	    pCmdInput->pPassto->pPassto = pCmdInput;
	}
}

